"""
========================================
🎭 VISUALモジュール - 動的pgzhelper対応版
========================================

pygame-zeroの標準Actorクラスをベースとし、pgzhelperが利用可能な場合はその機能も使用できるモジュールです。
透明化・モザイク効果・色効果・回転機能・明るさ効果・拡大機能を統合したモジュールです。

🔧 動的pgzhelper対応版の特徴:
- pgzhelperモジュールなしでも基本動作 ✅
- pgzhelperが利用可能な場合はその機能も使用可能 ✅
- 全ての既存機能を完全保持 ✅
- 既存コードの互換性100%保証 ✅

📋 基本的な使用方法:
    from visual import VisualEffect
    
    # 全エフェクト付きActorを作成
    boon = VisualEffect("boon", (400, 300))
    
    def draw():
        boon.draw(screen)  # エフェクト付き自動描画
    
    def update():
        boon.process_animation()  # アニメーション処理

========================================
📂 主要プロパティ一覧（使い方は全く同じ）
========================================

🔹 Actor継承プロパティ（そのまま使用可能）

    .x                       - X座標位置
    .y                       - Y座標位置
    .pos                     - 位置座標 (x, y)
    .visible                 - 表示状態フラグ
    .image                   - 画像ファイル名
    .angle                   - 回転角度（度数、0-359）

🔹 pgzhelper拡張プロパティ（pgzhelperが利用可能な場合のみ）

    .flip_x                  - 左右反転フラグ（pgzhelperがある場合のみ）
    .flip_y                  - 上下反転フラグ（pgzhelperがある場合のみ）
    .scale                   - スケール（pgzhelperがある場合のみ）

🔹 透明度関連プロパティ

    .alpha_level             - 現在の透明度（0-255）
    .alpha_increment         - 透明度の変化量（デフォルト10）
    .alpha_stop_at           - 透明度停止レベル（0-100%、デフォルト100）
    .is_alpha_on            - 透明度エフェクトのオン/オフ
    .animation_type         - アニメーションタイプ（"fade_out"/"fade_in"）

🔹 モザイク関連プロパティ

    .custom_pixel_size      - 現在のピクセルサイズ（初期値0）
    .pixel_increment        - 1段階あたりのピクセル増加量（デフォルト5）
    .current_mosaic_strength - 現在のモザイク強度（0-100）
    .is_mozaic_on           - モザイクエフェクトのオン/オフ
    .mozaic_stop_at         - モザイク停止レベル（％、デフォルト100）
    .max_pixel_size         - 最大ピクセルサイズ（自動計算）

🔹 色効果関連プロパティ

    .hue_shift              - 色相シフト値（0-199、200で一周）
    .base_bg_color          - 基本背景色（RGB タプル）
    .current_bg_color       - 現在の背景色（RGB タプル）
    .is_color_on            - 色効果のオン/オフ

🔹 明るさ関連プロパティ

    .brightness             - 明るさ値（-100～100）
    .is_brightness_on       - 明るさエフェクトのオン/オフ

🔹 拡大関連プロパティ

    .scale_factor           - 拡大倍率（1.0=等倍）
    .scale_increment        - 拡大変化量（デフォルト0.05）
    .scale_target           - 拡大目標倍率（デフォルト3.0）
    .is_scaling             - 拡大アニメーション実行中

🔹 アニメーション関連プロパティ

    .animation_running      - アニメーション実行中フラグ
    .animation_speed        - アニメーション速度（フレーム数、デフォルト5）
    .animation_counter      - アニメーションカウンター

========================================
🛠️ メソッド一覧（使い方は全く同じ）
========================================

🔸 初期化・設定メソッド

    VisualEffect(image, pos=(0, 0))
        説明: 全エフェクト付きActorを作成
        例: boon = VisualEffect("boon", (400, 300))

    .reset_effects()
        説明: 全てのエフェクトをリセット
        例: boon.reset_effects()

    .setup_image()
        説明: 画像を内部システムに設定・更新
        例: boon.image = "new_image"; boon.setup_image()

🔸 透明・モザイクアニメーションメソッド

    .alpha_animation()
        説明: フェードアウトアニメーション開始（不透明→透明）
        例: boon.alpha_animation()

    .fade_in_animation()
        説明: フェードインアニメーション開始（透明→不透明）
        例: boon.fade_in_animation()

    .mozaic_animation()
        説明: モザイクアニメーション開始
        例: boon.mozaic_animation()

    .process_animation()
        説明: アニメーション処理実行
        戻り値: "finished"=完了、True=継続中、None=未実行
        例: 
            result = boon.process_animation()
            if result == "finished":
                print("アニメーション完了！")

🔸 拡大アニメーションメソッド

    .scale_animation(increment=0.05, target=3.0)
        説明: 拡大アニメーション開始
        パラメータ: increment=変化量、target=目標倍率
        例: 
            boon.scale_animation()                    # デフォルト
            boon.scale_animation(0.1, 5.0)           # 高速で5倍

🔸 即座効果メソッド

    .set_alpha(alpha_value)
        説明: 透明度を即座に設定（0-255）
        例: 
            boon.set_alpha(128)    # 半透明
            boon.set_alpha(0)      # 完全透明

    .set_mosaic(strength)
        説明: モザイクの強さを即座に設定（0-100）
        例: 
            boon.set_mosaic(50)    # 中程度
            boon.set_mosaic(100)   # 最大

    .set_scale(scale_value)
        説明: 拡大倍率を即座に設定
        例: 
            boon.set_scale(2.0)    # 2倍に拡大
            boon.set_scale(1.0)    # 等倍に戻す

🔸 色効果メソッド

    .change_color(amount=25)
        説明: 色相を指定量変更（Scratchと同じ）
        例: 
            boon.change_color(20)    # 色相を20進める

    .change_bg_color(amount=25)
        説明: 背景色の色相を変更
        例: boon.change_bg_color(50)

    .set_base_bg_color(color)
        説明: 背景色の基本色を設定
        例: boon.set_base_bg_color((0, 100, 200))

    .apply_bg_color(screen)
        説明: 現在の背景色でスクリーンを塗りつぶす
        例: boon.apply_bg_color(screen)

    .reset_color()
        説明: 色効果のみをリセット
        例: boon.reset_color()

    .get_current_color_effect()
        説明: 現在の色相シフト値を取得（0-199）
        例: current_shift = boon.get_current_color_effect()

    .get_bg_color()
        説明: 現在の背景色を取得（RGB タプル）
        例: current_color = boon.get_bg_color()

🔸 回転・移動メソッド

    .move_to(x, y)
        説明: 指定座標に移動
        例: boon.move_to(400, 300)

    .move_random(x_min=50, x_max=750, y_min=50, y_max=540)
        説明: ランダムな場所に移動
        例: boon.move_random()

    .move_forward(distance)
        説明: 現在の角度方向に指定距離移動
        例: boon.move_forward(10)

🔸 pgzhelper拡張メソッド（pgzhelperが利用可能な場合のみ）

    .collide_pixel(other)
        説明: ピクセル単位の当たり判定（pgzhelperが必要）
        例: if boon.collide_pixel(arrow):

    .collidepoint_pixel(pos)
        説明: 点とのピクセル単位当たり判定（pgzhelperが必要）
        例: if boon.collidepoint_pixel(pos):

    .distance_to(target)
        説明: 距離を計算（pgzhelperが必要）
        例: dist = boon.distance_to(arrow)

    .direction_to(target)
        説明: 方向を計算（pgzhelperが必要）
        例: angle = boon.direction_to(arrow)

🔸 明るさ効果メソッド

    .change_brightness(amount)
        説明: 明るさを指定量変更（Scratchと同じ）
        例: boon.change_brightness(30)

    .set_brightness(value)
        説明: 明るさを指定値に設定（-100～100）
        例: boon.set_brightness(50)

    .reset_brightness()
        説明: 明るさをリセット
        例: boon.reset_brightness()

    .get_current_brightness()
        説明: 現在の明るさ値を取得
        例: current_brightness = boon.get_current_brightness()

🔸 描画メソッド

    .draw(screen=None)
        説明: 全エフェクト付き画像を自動描画
        例: boon.draw(screen)

========================================
🎪 実践的な使用例（動的対応版）
========================================

# 基本的な使用パターン（pgzhelperなしでも動作）
def basic_usage():
    boon = VisualEffect("boon", (400, 300))
    
    # 即座効果（常に使用可能）
    boon.set_alpha(128)         # 半透明
    boon.set_scale(1.5)         # 1.5倍に拡大
    boon.set_mosaic(30)         # 30%モザイク
    boon.change_color(50)       # 色相変更
    boon.change_brightness(20)  # 明るさ変更
    boon.angle = 45             # 45度回転
    
    # アニメーション効果（常に使用可能）
    boon.scale_animation(0.1, 3.0)  # 拡大アニメーション
    boon.alpha_animation()           # 透明アニメーション
    
    # リセット（常に使用可能）
    boon.reset_effects()        # 全効果リセット

# pgzhelper機能使用例（pgzhelperがある場合のみ）
def pgzhelper_usage():
    # pgzhelperが利用可能な場合の高精度当たり判定
    if hasattr(boon, 'collide_pixel'):
        if boon.collide_pixel(arrow):  # ピクセル単位判定
            print("精密な当たり判定！")
    else:
        if boon.colliderect(arrow):    # 標準の矩形判定
            print("標準の当たり判定！")

========================================
🚀 Scratchとの対応表（使い方は全く同じ）
========================================

Scratch命令                    → VisualEffect対応メソッド
「色の効果を25ずつ変える」      → boon.change_color(25)
「明るさの効果を30ずつ変える」   → boon.change_brightness(30)
「明るさの効果を-50にする」    → boon.set_brightness(-50)
「透明度を50%にする」          → boon.set_alpha(127)
「大きさを200%にする」         → boon.set_scale(2.0)
「15度右に回す」              → boon.angle += 15
「45度に向ける」              → boon.angle = 45
「画像効果をなくす」           → boon.reset_effects()
「10歩動かす」                → boon.move_forward(10)

========================================
⚠️ 重要な注意事項（動的対応版）
========================================

1. 基本機能（常に使用可能）
    ○ エフェクト機能全般
    ○ 基本的な移動・回転
    ○ 標準の当たり判定

2. 拡張機能（pgzhelperが必要）
    ○ ピクセル単位当たり判定
    ○ 高精度な移動・回転機能
    ○ 距離・方向計算

3. 動作モード
    - pgzhelperなし: 基本機能のみ
    - pgzhelperあり: 全機能使用可能

========================================
🏆 動的pgzhelper対応版のメリット
========================================

✅ 柔軟な対応

1. **段階的導入**
    - 基本機能から始められる
    - 必要に応じてpgzhelper追加

2. **完全な互換性**
    - 既存コードがそのまま動作
    - 全ての機能を完全保持

3. **環境に応じた最適化**
    - pgzhelperがあれば高機能
    - なくても基本動作保証

========================================
🎓 まとめ（動的pgzhelper対応版）
========================================

動的pgzhelper対応版visualモジュールは、環境に応じて最適な機能を提供する
柔軟性の高いビジュアルエフェクトツールです。

主な特徴:
- 🎭 環境に応じた動的機能提供
- 🔧 pgzhelperなしでも基本動作
- ⚡ pgzhelperがあれば拡張機能も使用可能
- 🎯 既存コードの完全互換性
- 🛡️ 段階的な機能導入が可能

これにより、あらゆる環境で安心して使用できる
究極のビジュアルエフェクトモジュールが完成しました。

========================================
"""

import pygame
import random
import math

# pgzhelperの動的インポート
try:
    from pgzhelper import Actor as PgzhelperActor
    PGZHELPER_AVAILABLE = True
    print("pgzhelper detected: 拡張機能が利用可能です")
except ImportError:
    from pgzero.actor import Actor as PgzhelperActor
    PGZHELPER_AVAILABLE = False
    print("pgzhelper not found: 基本機能のみ利用可能です")

class VisualEffect(PgzhelperActor):
    
    def _inherit_say_functionality(self):
        try:
            from say import text_display
            
            def say(text, seconds=None, size=40, color="black", x_offset=None, y_offset=None, fontname=None, anchor="center"):
                text_display.say(self, text, seconds, size, color, x_offset, y_offset, fontname, anchor)
            
            def is_talking():
                return text_display.is_talking(self)
            
            def clear_text():
                text_display.clear(self)
            
            self.say = say
            self.is_talking = is_talking
            self.clear_text = clear_text
            
        except ImportError:
            pass
    
    def __init__(self, *args, **kwargs):
        if len(args) == 1:
            super().__init__(args[0], (0, 0))
        elif len(args) == 2:
            super().__init__(args[0], args[1])
        else:
            super().__init__(*args, **kwargs)
        
        self.visible = True
        
        # 回転属性
        self.angle = 0.0
        self._angle = 0.0
        
        # 明るさ属性
        self.brightness = 0
        self.is_brightness_on = False
        
        # 拡大属性
        self.scale_factor = 1.0
        self.scale_increment = 0.05
        self.scale_target = 3.0
        self.is_scaling = False
        
        # sayモジュール機能継承
        self._inherit_say_functionality()
        
        # エフェクト用属性
        self.original_image = None
        self.processed_image = None
        self.current_looks_actor = None
        
        # 透明度関連
        self.alpha_level = 255
        self.alpha_increment = 10
        self.alpha_stop_at = 100
        self.is_alpha_on = False
        
        # モザイク関連
        self.is_pixelated = False
        self.pixelation_level = 0
        self.pixel_increment = 5
        self.custom_pixel_size = 0
        self.is_mozaic_on = False
        self.mozaic_stop_at = 100
        self.current_mosaic_strength = 0
        
        # 色効果関連
        self.hue_shift = 0
        self.base_bg_color = (255, 255, 255)
        self.current_bg_color = self.base_bg_color
        self.is_color_on = False
        
        # アニメーション関連
        self.animation_type = "fade_out"
        self.animation_running = False
        self.animation_speed = 5
        self.animation_counter = 0
        
        # 表示状態
        self.is_visible = True
        self.is_actor_draw = True
        
        # 初期化
        self.setup_image()

    def setup_image(self):
        """画像を内部システムに設定・更新する"""
        if hasattr(self, '_surf') and self._surf:
            self.original_image = self._surf.copy()
            self.processed_image = self._surf.copy()
            width, height = self._surf.get_size()
            self.max_pixel_size = min(width, height)

    def _update_current_drawable(self):
        has_effects = (self.is_mozaic_on or self.is_alpha_on or 
                      self.is_color_on or self.is_brightness_on or 
                      self.scale_factor != 1.0)
        
        if has_effects and self.processed_image:
            self.current_looks_actor = self.processed_image
            self._angle = self.angle
        else:
            self.current_looks_actor = None
            self._angle = self.angle

    def _update_draw_position(self):
        if self.processed_image:
            self.draw_pos = (
                self.x - self.processed_image.get_width() // 2,
                self.y - self.processed_image.get_height() // 2
            )

    def move_to(self, x, y):
        """指定座標に移動"""
        self.x = x
        self.y = y

    def move_random(self, x_min=50, x_max=750, y_min=50, y_max=540):
        """ランダムな場所に移動"""
        self.x = random.randint(x_min, x_max)
        self.y = random.randint(y_min, y_max)
        return (self.x, self.y)
    
    def move_forward(self, distance):
        """現在の角度方向に指定距離移動"""
        if PGZHELPER_AVAILABLE and hasattr(super(), 'move_forward'):
            # pgzhelperのmove_forwardを使用
            super().move_forward(distance)
        else:
            # 標準的な実装
            angle_rad = math.radians(self.angle)
            self.x += distance * math.cos(angle_rad)
            self.y -= distance * math.sin(angle_rad)

    def scale_animation(self, increment=0.05, target=3.0):
        """拡大アニメーションを開始"""
        if not self.original_image:
            return False
        
        self.scale_increment = increment
        self.scale_target = target
        self.is_scaling = True
        return True

    def change_brightness(self, amount):
        """明るさを変更する (Scratchの「明るさの効果を○ずつ変える」に相当)"""
        self.brightness = max(-100, min(100, self.brightness + amount))
        self.is_brightness_on = True if self.brightness != 0 else False
        self._apply_all_effects()
        return self.brightness
        
    def set_brightness(self, value):
        """明るさを設定する (Scratchの「明るさの効果を○にする」に相当)"""
        self.brightness = max(-100, min(100, value))
        self.is_brightness_on = True if self.brightness != 0 else False
        self._apply_all_effects()
        return self.brightness
        
    def reset_brightness(self):
        """明るさをリセットする (Scratchの「画像効果をなくす」に相当)"""
        self.brightness = 0
        self.is_brightness_on = False
        self._apply_all_effects()
        
    def get_current_brightness(self):
        """現在の明るさ値を取得する"""
        return self.brightness

    def _apply_brightness_effect(self, surface):
        """明るさ効果を適用する"""
        if self.brightness == 0:
            return surface
            
        width, height = surface.get_size()
        new_surf = pygame.Surface((width, height), pygame.SRCALPHA)
        
        if self.brightness > 0:
            factor = self.brightness / 100
            for y in range(height):
                for x in range(width):
                    try:
                        pixel = surface.get_at((x, y))
                        r, g, b, a = pixel
                        if a > 0:
                            r = min(255, int(r + (255 - r) * factor))
                            g = min(255, int(g + (255 - g) * factor))
                            b = min(255, int(b + (255 - b) * factor))
                            new_surf.set_at((x, y), (r, g, b, a))
                        else:
                            new_surf.set_at((x, y), pixel)
                    except (IndexError, ValueError):
                        continue
        else:
            factor = -self.brightness / 100
            for y in range(height):
                for x in range(width):
                    try:
                        pixel = surface.get_at((x, y))
                        r, g, b, a = pixel
                        if a > 0:
                            r = max(0, int(r * (1 - factor)))
                            g = max(0, int(g * (1 - factor)))
                            b = max(0, int(b * (1 - factor)))
                            new_surf.set_at((x, y), (r, g, b, a))
                        else:
                            new_surf.set_at((x, y), pixel)
                    except (IndexError, ValueError):
                        continue
        
        return new_surf

    def change_color(self, amount=25):
        """このアクターの色相を指定量変更する（Scratchと同じ）"""
        self.hue_shift = (self.hue_shift + amount) % 200
        self.is_color_on = True if self.hue_shift != 0 else False
        self._apply_all_effects()
        return self.hue_shift
    
    def change_bg_color(self, amount=25):
        """背景色の色相を変更する"""
        self.hue_shift = (self.hue_shift + amount) % 200
        self.is_color_on = True if self.hue_shift != 0 else False
        self._update_bg_color()
        return self.hue_shift
    
    def set_base_bg_color(self, color):
        """背景色の基本色を設定する"""
        self.base_bg_color = color
        self.current_bg_color = color
    
    def _update_bg_color(self):
        """現在の色相シフト値に基づいて背景色を更新する"""
        if self.hue_shift != 0:
            r, g, b = self.base_bg_color
            h, s, v = self._rgb_to_hsv(r/255, g/255, b/255)
            h = (h + self.hue_shift / 200) % 1.0
            if s < 0.3:
                s = 0.7
            r, g, b = self._hsv_to_rgb(h, s, v)
            self.current_bg_color = (int(r * 255), int(g * 255), int(b * 255))
        else:
            self.current_bg_color = self.base_bg_color
    
    def get_bg_color(self):
        """現在の背景色を取得する"""
        return self.current_bg_color
    
    def apply_bg_color(self, screen):
        """現在の背景色でスクリーンを塗りつぶす"""
        self._update_bg_color()
        screen.fill(self.current_bg_color)
    
    def reset_color(self):
        """色効果のみをリセットする"""
        self.hue_shift = 0
        self.current_bg_color = self.base_bg_color
        self.is_color_on = False
        self._apply_all_effects()
    
    def get_current_color_effect(self):
        """現在の色効果値を取得する"""
        return self.hue_shift

    def reset_effects(self):
        """全てのエフェクト（透明・モザイク・色・回転・明るさ・拡大）を元の状態に戻す"""
        if self.original_image:
            self.processed_image = self.original_image.copy()
            self.is_pixelated = False
            self.pixelation_level = 0
            self.is_visible = True
            self.custom_pixel_size = 0
            self.alpha_level = 255
            self.animation_running = False
            self.animation_counter = 0
            self.is_mozaic_on = False
            self.is_alpha_on = False
            self.hue_shift = 0
            self.is_color_on = False
            self.current_bg_color = self.base_bg_color
            self.angle = 0.0
            self.brightness = 0
            self.is_brightness_on = False
            self.scale_factor = 1.0
            self.is_scaling = False
            self.pixelation_level = 0
            self.current_mosaic_strength = 0
            self.is_actor_draw = True
            self.animation_type = "fade_out"
            self.visible = True
            self.current_looks_actor = None
            self._update_current_drawable()

    def _apply_all_effects(self):
        """全てのエフェクト（色・モザイク・透明・明るさ・拡大）を適用する"""
        if not self.original_image:
            return
            
        base_image = self.original_image.copy()
        
        if self.is_color_on and self.hue_shift != 0:
            base_image = self._apply_color_shift(base_image)
        
        if self.is_brightness_on and self.brightness != 0:
            base_image = self._apply_brightness_effect(base_image)
        
        if self.is_mozaic_on and not self.animation_running:
            pixel_size = max(1, int(self.max_pixel_size * (self.current_mosaic_strength / 100)))
            base_image = self.pixelate(base_image, pixel_size)
        
        if self.scale_factor != 1.0:
            original_size = base_image.get_size()
            new_width = int(original_size[0] * self.scale_factor)
            new_height = int(original_size[1] * self.scale_factor)
            base_image = pygame.transform.scale(base_image, (new_width, new_height))
        
        if self.is_alpha_on and not self.animation_running:
            base_image.set_alpha(self.alpha_level)
        
        self.processed_image = base_image
        self._update_current_drawable()

    def _apply_color_shift(self, surface):
        """色相シフトを適用する"""
        if self.hue_shift == 0:
            return surface
            
        shifted = pygame.Surface(surface.get_size(), pygame.SRCALPHA)
        
        hue_factor = self.hue_shift / 200
        for y in range(surface.get_height()):
            for x in range(surface.get_width()):
                r, g, b, a = surface.get_at((x, y))
                if a > 0:
                    h, s, v = self._rgb_to_hsv(r / 255, g / 255, b / 255)
                    h = (h + hue_factor) % 1.0
                    r, g, b = self._hsv_to_rgb(h, s, v)
                    shifted.set_at((x, y), (int(r * 255), int(g * 255), int(b * 255), a))
                else:
                    shifted.set_at((x, y), (r, g, b, a))
        
        return shifted

    def mozaic_animation(self):
        """モザイクアニメーションを開始する"""
        if not self.original_image or not self.is_visible:
            return False
            
        self.is_mozaic_on = True
        
        if not self.animation_running:
            self.start_animation()
        
        return True

    def alpha_animation(self):
        """透明度アニメーションを開始する（不透明→透明）"""
        if not self.original_image or not self.is_visible:
            return
            
        self.is_alpha_on = True
        self.animation_type = "fade_out"
        
        if not self.animation_running:
            self.start_animation()

    def set_mosaic(self, strength):
        """モザイクの強さを保存して即座に設定"""
        strength = max(0, min(100, int(strength)))
        self.current_mosaic_strength = strength
        
        if strength > 0:
            self.is_mozaic_on = True
        else:
            self.is_mozaic_on = False
        
        self._apply_all_effects()
        return strength

    def fade_in_animation(self):
        """透明度アニメーションを開始する（透明→不透明）"""
        if not self.original_image or not self.is_visible:
            return False
            
        self.is_alpha_on = True
        self.animation_type = "fade_in"
        
        if not self.animation_running:
            self.animation_running = True
            self.pixelation_level = 1
            self.custom_pixel_size = 5
            self.alpha_level = 0
            self.animation_counter = 0
            
            if self.original_image:
                self.processed_image = self.original_image.copy()
                self._apply_all_effects()
                
            self._update_current_drawable()
            self.visible = False
        
        return True
        
    def start_animation(self):
        """アニメーションの共通処理を開始する"""
        self.animation_running = True
        self.pixelation_level = 1
        self.custom_pixel_size = 5
        
        if self.animation_type == "fade_in":
            self.alpha_level = 0
        else:
            self.alpha_level = 255
            
        self.animation_counter = 0
        
        if self.original_image:
            self.processed_image = self.original_image.copy()
            self._apply_all_effects()
            
        self._update_current_drawable()
        self.visible = False

    def process_animation(self):
        """アニメーション処理を行う関数（メインループから呼び出す）"""
        result = None
        alpha_mozaic_finished = False

        scale_finished = False
        if self.is_scaling and self.scale_factor < self.scale_target:
            saved_angle = self.angle
            self.scale_factor += self.scale_increment
            
            if self.scale_factor >= self.scale_target:
                self.scale_factor = self.scale_target
                self.is_scaling = False
                scale_finished = True
            
            self.angle = saved_angle
            self._angle = saved_angle

        if self.animation_running:
            self.animation_counter += 1

            if self.animation_counter % self.animation_speed == 0:
                result = self.update_animation()
                
                if result == "finished":
                    alpha_mozaic_finished = True
                    self.is_mozaic_on = False
                    self.is_alpha_on = False
                    
                    if self.animation_type == "fade_out":
                        self.is_actor_draw = False
                        self.visible = False
                    else:
                        self.is_actor_draw = True
                        self.visible = True
                    
                    self.current_looks_actor = None
                    self._update_current_drawable()

        if alpha_mozaic_finished or scale_finished:
            self.scale_factor = 1.0
            return "finished"
        else:
            return result

    def update_animation(self):
        """アニメーションを更新する"""
        if not self.animation_running:
            return False

        if self.original_image:
            self.processed_image = self.original_image.copy()
        
        if self.is_mozaic_on:
            self.pixelation_level += 1
            self.custom_pixel_size = self.pixel_increment * self.pixelation_level
            self.processed_image = self.pixelate(self.processed_image, self.custom_pixel_size)
        
        if self.is_color_on and self.hue_shift != 0:
            self.processed_image = self._apply_color_shift(self.processed_image)
        
        if self.is_brightness_on and self.brightness != 0:
            self.processed_image = self._apply_brightness_effect(self.processed_image)
        
        if self.scale_factor != 1.0:
            original_size = self.processed_image.get_size()
            new_width = int(original_size[0] * self.scale_factor)
            new_height = int(original_size[1] * self.scale_factor)
            self.processed_image = pygame.transform.scale(self.processed_image, (new_width, new_height))
        
        if self.is_alpha_on:
            if self.animation_type == "fade_out":
                target_alpha = int(255 * (100 - self.alpha_stop_at) / 100)
                self.alpha_level = max(target_alpha, self.alpha_level - self.alpha_increment)
            else:
                self.alpha_level = min(255, self.alpha_level + self.alpha_increment)
            
            self.processed_image.set_alpha(self.alpha_level)
        
        self._update_draw_position()
        self._update_current_drawable()
        
        alpha_finished = False
        mozaic_finished = False
        
        if self.is_alpha_on:
            if self.animation_type == "fade_out":
                target_alpha = int(255 * (100 - self.alpha_stop_at) / 100)
                alpha_finished = (self.alpha_level <= target_alpha)
            else:
                alpha_finished = (self.alpha_level >= 255)
        
        if self.is_mozaic_on:
            current_mozaic_percent = int(self.custom_pixel_size * 100 / self.max_pixel_size)
            mozaic_finished = (current_mozaic_percent >= self.mozaic_stop_at)
        
        is_finished = alpha_finished or mozaic_finished
        
        if is_finished:
            self.animation_running = False
            if self.animation_type == "fade_out":
                self.is_visible = False
            return "finished"

        return True

    def pixelate(self, surface, pixel_size):
        """ピクセル化を適用する関数"""
        try:
            pixelated_surface = pygame.Surface(surface.get_size(), pygame.SRCALPHA)
            width, height = surface.get_size()

            for y in range(0, height, pixel_size):
                for x in range(0, width, pixel_size):
                    block_width = min(pixel_size, width - x)
                    block_height = min(pixel_size, height - y)

                    if block_width <= 0 or block_height <= 0:
                        continue

                    rect = pygame.Rect(x, y, block_width, block_height)
                    sub_surface = surface.subsurface(rect)
                    avg_color = pygame.transform.average_color(sub_surface)
                    pixelated_surface.fill(avg_color, rect=(x, y, block_width, block_height))

            return pixelated_surface
        except Exception as e:
            return surface.copy()

    def set_alpha(self, alpha_value):
        """透明度を即座に設定 (0-255)"""
        self.alpha_level = max(0, min(255, int(alpha_value)))
        if self.alpha_level < 255:
            self.is_alpha_on = True
        else:
            self.is_alpha_on = False
        self._apply_all_effects()
        if self.is_alpha_on:
            self.visible = False
        elif not (self.is_mozaic_on or self.is_color_on or self.is_brightness_on):
            self.visible = True
    
    def set_ghost(self, alpha_value):
        """透明度を即座に設定 (0-255) - 互換性のための別名"""
        return self.set_alpha(alpha_value)

    def set_scale(self, scale_value):
        """拡大倍率を即座に設定"""
        self.scale_factor = max(0.1, scale_value)
        self._apply_all_effects()
        return self.scale_factor
    
    def draw(self, screen=None):
        """全エフェクト（透明・モザイク・色・回転・明るさ・拡大）付き描画"""
        if self.is_mozaic_on or self.is_alpha_on or self.is_color_on or self.is_brightness_on or self.scale_factor != 1.0:
            if self.processed_image:
                image_to_draw = self.processed_image
                
                # pgzhelperが利用可能で反転機能があるかチェック
                if PGZHELPER_AVAILABLE and hasattr(self, 'flip_x') and hasattr(self, 'flip_y'):
                    if self.flip_x:
                        image_to_draw = pygame.transform.flip(image_to_draw, True, False)
                    if self.flip_y:
                        image_to_draw = pygame.transform.flip(image_to_draw, False, True)
                
                if self.angle != 0:
                    image_to_draw = pygame.transform.rotate(image_to_draw, self.angle)
                
                draw_x = self.x - image_to_draw.get_width() // 2
                draw_y = self.y - image_to_draw.get_height() // 2
                
                if screen:
                    screen.blit(image_to_draw, (draw_x, draw_y))
                else:
                    import __main__
                    if hasattr(__main__, 'screen'):
                        __main__.screen.blit(image_to_draw, (draw_x, draw_y))
            else:
                if self.visible:
                    super().draw()
        else:
            if self.visible:
                super().draw()

    def _rgb_to_hsv(self, r, g, b):
        """RGB to HSV変換"""
        mx = max(r, g, b)
        mn = min(r, g, b)
        df = mx - mn
        if mx == mn:
            h = 0
        elif mx == r:
            h = (60 * ((g - b) / df) + 360) % 360
        elif mx == g:
            h = (60 * ((b - r) / df) + 120) % 360
        elif mx == b:
            h = (60 * ((r - g) / df) + 240) % 360
        s = 0 if mx == 0 else df / mx
        v = mx
        return h / 360, s, v
    
    def _hsv_to_rgb(self, h, s, v):
        """HSV to RGB変換"""
        if s == 0.0:
            return v, v, v
        i = int(h * 6)
        f = (h * 6) - i
        p = v * (1 - s)
        q = v * (1 - s * f)
        t = v * (1 - s * (1 - f))
        i %= 6
        if i == 0: return v, t, p
        if i == 1: return q, v, p
        if i == 2: return p, v, t
        if i == 3: return p, q, v
        if i == 4: return t, p, v
        if i == 5: return v, p, q

    def setup_visual_costume(self):
        """costumeモジュールとの連携用メソッド：画像変更時にvisualエフェクトを同期"""
        if hasattr(self, '_surf') and self._surf:
            # すべての状態を完全保存
            saved_angle = getattr(self, 'angle', 0.0)
            saved_pos = (self.x, self.y)
            
            # pgzhelper固有の属性も保存（利用可能な場合のみ）
            saved_flip_x = getattr(self, 'flip_x', False) if PGZHELPER_AVAILABLE else False
            saved_flip_y = getattr(self, 'flip_y', False) if PGZHELPER_AVAILABLE else False
            
            # エフェクト状態を保存
            current_mosaic_strength = self.current_mosaic_strength
            current_alpha_level = self.alpha_level
            current_alpha_on = self.is_alpha_on
            current_color_on = self.is_color_on
            current_hue_shift = self.hue_shift
            current_brightness = self.brightness
            current_brightness_on = self.is_brightness_on
            current_scale_factor = self.scale_factor
            current_is_scaling = self.is_scaling
            
            # アニメーション状態も保存
            current_animation_running = self.animation_running
            current_animation_type = self.animation_type
            current_animation_counter = self.animation_counter
            
            # 内部画像を更新
            self.setup_image()
            
            # すべての状態を完全復元
            self.angle = saved_angle
            self._angle = saved_angle
            self.x, self.y = saved_pos
            
            # pgzhelper固有の属性も復元（利用可能な場合のみ）
            if PGZHELPER_AVAILABLE:
                if hasattr(self, 'flip_x'):
                    self.flip_x = saved_flip_x
                if hasattr(self, 'flip_y'):
                    self.flip_y = saved_flip_y
            
            # エフェクト状態を復元
            self.current_mosaic_strength = current_mosaic_strength
            self.alpha_level = current_alpha_level
            self.is_alpha_on = current_alpha_on
            self.is_color_on = current_color_on
            self.hue_shift = current_hue_shift
            self.brightness = current_brightness
            self.is_brightness_on = current_brightness_on
            self.scale_factor = current_scale_factor
            self.is_scaling = current_is_scaling
            
            # アニメーション状態も復元
            self.animation_running = current_animation_running
            self.animation_type = current_animation_type
            self.animation_counter = current_animation_counter
            
            # エフェクトを新しい画像に再適用
            if (self.current_mosaic_strength > 0 or self.is_alpha_on or 
                self.is_color_on or self.is_brightness_on or 
                self.scale_factor != 1.0):
                self._apply_all_effects()
            
            # 描画対象を更新
            self._update_current_drawable()
